# -*- coding: utf-8 -*-
import numpy as np
import os

from lutio.format import LUT3DFormat

__all__ = ['Pandora3DLUTFormat', 'AppleColor3DLUTFormat']

#
# Helper

PANDORA3DLUT_TEMPLATE = """#HEADER
#filename: {stem}
#type: Pandora 3D LUT file
#format: 1.00
#title: n/a

#END

channel 3d
in {flatsize}
out {bitrange}

format lut

values\tred\tgreen\tblue
{ivalues}
"""


#
# Master

class Pandora3DLUTFormat(LUT3DFormat):
    class Meta:
        name = "Pandora 3D M3D"
        extensions = (".m3d",)

    metakey = 'pandora'

    supported_sizes = (17, 33)
    supported_bitdepths = (16,)

    def __init__(self, size=17, bitdepth=16):
        super().__init__(size, bitdepth)

    def to_file(self, lut, filename):
        lut, filename = self._prepare_to_file_(lut, filename)

        lut = lut.reshape(-1, 3)
        lut = np.c_[np.arange(lut.shape[0]), lut]

        values = self._lut_to_str_(lut, fmt="%d")

        with open(filename, 'w') as f:
            stem = os.path.splitext(os.path.basename(filename))[0]

            f.write(PANDORA3DLUT_TEMPLATE.format(stem=stem,
                                                 flatsize=self.size ** 3,
                                                 bitrange=2 ** self.bitdepth,
                                                 ivalues=values))


class AppleColor3DLUTFormat(Pandora3DLUTFormat):
    class Meta:
        name = "Apple Color"
        extensions = (".mga",)

    metakey = 'applecolor'
