# -*- coding: utf-8 -*-
import numpy as np
import os

from lutio.format import LUT3DFormat

__all__ = ['NucodaLUT3DFormat', 'CortexDailiesLUT3DFormat']

#
# Helper

NUCODALUT3D_TEMPLATE = """NUCODA_3D_CUBE 1

TITLE "{stem}"

LUT_1D_SIZE 4096
LUT_3D_SIZE {size}

{grid}

{values}
"""


#
# Master

class NucodaLUT3DFormat(LUT3DFormat):
    class Meta:
        extensions = (".cms",)
        name = "Digital Vision Nucoda"

    metakey = 'nucoda'

    supported_sizes = (17, 33, 65)

    def __init__(self, size=33):
        super().__init__(size)

    def to_file(self, lut, filename):
        lut, filename = self._prepare_to_file_(lut, filename)

        lut = lut.transpose(2, 1, 0, 3)

        grid_x = np.linspace(0, self.size, num=4096, endpoint=False)
        grid = self._lut_to_str_(np.c_[grid_x, grid_x, grid_x])

        values = self._lut_to_str_(lut)

        with open(filename, 'w') as f:
            stem = os.path.splitext(os.path.basename(filename))[0]

            f.write(NUCODALUT3D_TEMPLATE.format(size=self.size,
                                                stem=stem,
                                                grid=grid,
                                                values=values))


class CortexDailiesLUT3DFormat(NucodaLUT3DFormat):
    class Meta:
        extensions = (".cms",)
        name = "CORTEX Dailies"

    metakey = 'cortex'
