# -*- coding: utf-8 -*-
import math

from lutio.format import LUT3DFormat

__all__ = ['Lustre3DLUTFormat', 'Silverstack3DLUTFormat']

#
# Helper

LUSTRE3DLUT_TEMPLATE = """3DMESH
Mesh {inbits:.0f} {outbits:d}

{values}

LUT8
gamma {gamma:.1f}
"""


#
# Master

class Lustre3DLUTFormat(LUT3DFormat):
    class Meta:
        name = "Autodesk Lustre"
        extensions = (".lut",)

    metakey = 'lustre'

    supported_sizes = (17, 33, 65)
    supported_bitdepths = (10, 12, 14, 16)

    def __init__(self, size=33, bitdepth=14):
        super().__init__(size, bitdepth)

    def to_file(self, lut, filename):
        lut, filename = self._prepare_to_file_(lut, filename)
        values = self._lut_to_str_(lut, fmt="%d")

        with open(filename, 'w') as f:
            inbits = math.log2(self.size - 1)

            f.write(LUSTRE3DLUT_TEMPLATE.format(size=self.size,
                                                inbits=inbits,
                                                outbits=self.bitdepth,
                                                values=values,
                                                gamma=1))


class Silverstack3DLUTFormat(Lustre3DLUTFormat):
    class Meta:
        name = "Pomfort Silverstack"
        extensions = (".lut",)

    metakey = 'silverstack'
