# -*- coding: utf-8 -*-
import struct

from lutio.format import LUT3DFormat

__all__ = ['FlandersDITLUT3DFormat']


class FlandersDITLUT3DFormat(LUT3DFormat):
    class Meta:
        extensions = (".dat",)
        name = "Flanders Scientific DIT"

    metakey = 'flandersdit'

    supported_sizes = (17,)
    supported_bitdepths = (10,)

    def __init__(self, size=17, bitdepth=10):
        super().__init__(size, bitdepth)

    def to_file(self, lut, filename):
        lut, filename = self._prepare_to_file_(lut, filename)

        lut = lut.transpose(2, 1, 0, 3)

        with open(filename, 'wb') as f:
            lut = lut.reshape(-1, 3)
            lut = lut[:, 0] | (lut[:, 1] << 10) | (lut[:, 2] << 20)
            payload = lut.astype('>i').tobytes()

            header_struct = struct.Struct('< L L 16s 16s L L 16s 63s')
            header_values = (0x42340299,  # magic
                             0x02000000,  # format version
                             b"",         # model
                             b"0.1.00",   # payload version
                             self.compute_payload_checksum(payload),
                             19652,       # payload length
                             b"",         # description
                             b"")         # reseved

            header = header_struct.pack(*header_values)
            header += self.compute_header_checksum(header)

            f.write(header)
            f.write(payload)

    @staticmethod
    def compute_header_checksum(header):
        return struct.pack("B", sum(bytearray(header)) % (2 ** 8))

    @staticmethod
    def compute_payload_checksum(payload):
        return int(sum(bytearray(payload)))
