# -*- coding: utf-8 -*-
import numpy as np

from lutio.format import LUT3DFormat

__all__ = ['Autodesk3DLUTFormat', 'Blackmagic3DLUTFormat',
           'Cinetal3DLUTFormat', 'Scratch3DLUTFormat']

#
# Helper

THREEDL3DLUT_TEMPLATE = """{grid_x}
{values}
"""


#
# Master

class ThreeDL3DLUTFormat(LUT3DFormat):
    supported_sizes = (16, 17, 32, 33)
    supported_bitdepths = (10, 12, 16)

    def __init__(self, size=33, bitdepth=12):
        super().__init__(size, bitdepth)
        self.inbits = 10

    def to_file(self, lut, filename):
        lut, filename = self._prepare_to_file_(lut, filename)

        grid_x = np.ceil(np.linspace(0, 2 ** self.inbits - 1, self.size))
        grid_x = self._lut_to_str_(grid_x, fmt="%d", newline=" ")

        values = self._lut_to_str_(lut, fmt="%d")

        with open(filename, 'w') as f:
            f.write(THREEDL3DLUT_TEMPLATE.format(grid_x=grid_x,
                                                 values=values))


class Autodesk3DLUTFormat(ThreeDL3DLUTFormat):
    class Meta:
        name = "Autodesk"
        extensions = (".3dl",)

    metakey = 'autodesk'


class Blackmagic3DLUTFormat(ThreeDL3DLUTFormat):
    class Meta:
        name = "Blackmagic"
        extensions = (".3dl",)

    metakey = 'blackmagic'


class Cinetal3DLUTFormat(ThreeDL3DLUTFormat):
    class Meta:
        name = "Cine-tal"
        extensions = (".3dl",)

    metakey = 'cinetal'


class Scratch3DLUTFormat(ThreeDL3DLUTFormat):
    class Meta:
        name = "Assimilate Scratch"
        extensions = (".3dl",)

    metakey = 'scratch'
